"use client"

import type React from "react"

import { useState } from "react"
import { useRouter } from "next/navigation"
import { ArrowLeft, Globe, Upload, Check, X } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Checkbox } from "@/components/ui/checkbox"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { LogoWithText } from "@/components/logo"
import { useAuth } from "@/components/auth-provider"

export default function SubmitWebsitePage() {
  const router = useRouter()
  const { user, isLoading } = useAuth()
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [submitted, setSubmitted] = useState(false)
  const [error, setError] = useState("")

  const [formData, setFormData] = useState({
    title: "",
    url: "",
    imageUrl: "",
    supportImageCrawl: false,
    supportWebCrawl: true,
    ownerConfirmed: false,
  })

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()

    if (!formData.ownerConfirmed) {
      setError("Please confirm that you are the owner or developer of this website.")
      return
    }

    if (!formData.title.trim() || !formData.url.trim()) {
      setError("Please fill in all required fields.")
      return
    }

    // Basic URL validation
    try {
      new URL(formData.url)
    } catch {
      setError("Please enter a valid URL.")
      return
    }

    setIsSubmitting(true)
    setError("")

    try {
      const response = await fetch("/api/submit-website", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(formData),
      })

      if (!response.ok) {
        throw new Error("Failed to submit website")
      }

      setSubmitted(true)
    } catch (error) {
      console.error("Submission error:", error)
      setError("Failed to submit website. Please try again.")
    } finally {
      setIsSubmitting(false)
    }
  }

  const handleInputChange = (field: string, value: any) => {
    setFormData((prev) => ({ ...prev, [field]: value }))
    if (error) setError("")
  }

  if (isLoading) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto mb-4"></div>
          <p className="text-muted-foreground">Loading...</p>
        </div>
      </div>
    )
  }

  if (!user) {
    return (
      <div className="min-h-screen bg-background flex flex-col">
        <header className="flex items-center p-4 border-b border-border">
          <Button variant="ghost" size="sm" onClick={() => router.push("/")} className="mr-4">
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Search
          </Button>
          <LogoWithText size="sm" />
        </header>

        <main className="flex-1 flex items-center justify-center p-4">
          <Card className="w-full max-w-md">
            <CardContent className="pt-6 text-center">
              <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                <Globe className="w-8 h-8 text-primary" />
              </div>
              <h2 className="text-xl font-semibold mb-2">Authentication Required</h2>
              <p className="text-muted-foreground mb-6">You need to be logged in to submit a website to IndusSearch.</p>
              <div className="space-y-3">
                <Button onClick={() => router.push("/ads-center/auth")} className="w-full">
                  Sign In / Register
                </Button>
                <Button variant="outline" onClick={() => router.push("/")} className="w-full">
                  Back to Homepage
                </Button>
              </div>
            </CardContent>
          </Card>
        </main>
      </div>
    )
  }

  if (submitted) {
    return (
      <div className="min-h-screen bg-background flex flex-col">
        <header className="flex items-center p-4 border-b border-border">
          <Button variant="ghost" size="sm" onClick={() => router.push("/")} className="mr-4">
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Search
          </Button>
          <LogoWithText size="sm" />
        </header>

        <main className="flex-1 flex items-center justify-center p-4">
          <Card className="w-full max-w-md">
            <CardContent className="pt-6 text-center">
              <div className="w-16 h-16 bg-green-100 dark:bg-green-900/20 rounded-full flex items-center justify-center mx-auto mb-4">
                <Check className="w-8 h-8 text-green-600 dark:text-green-400" />
              </div>
              <h2 className="text-xl font-semibold mb-2">Website Submitted Successfully!</h2>
              <p className="text-muted-foreground mb-4">
                Thank you for submitting your website. Our team will review it and add it to our search index if
                approved.
              </p>
              <Button onClick={() => router.push("/")} className="w-full">
                Return to Homepage
              </Button>
            </CardContent>
          </Card>
        </main>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-background flex flex-col">
      <header className="flex items-center p-4 border-b border-border">
        <Button variant="ghost" size="sm" onClick={() => router.push("/")} className="mr-4">
          <ArrowLeft className="w-4 h-4 mr-2" />
          Back to Search
        </Button>
        <LogoWithText size="sm" />
      </header>

      <main className="flex-1 p-4">
        <div className="max-w-2xl mx-auto">
          <div className="text-center mb-8">
            <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
              <Globe className="w-8 h-8 text-primary" />
            </div>
            <h1 className="text-3xl font-bold mb-2">Submit Your Website</h1>
            <p className="text-muted-foreground">Add your website to IndusSearch to help users discover your content</p>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Website Information</CardTitle>
              <CardDescription>
                Please provide accurate information about your website. All submissions are reviewed before being added
                to our search index.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-6">
                {error && (
                  <Alert variant="destructive">
                    <X className="w-4 h-4" />
                    <AlertDescription>{error}</AlertDescription>
                  </Alert>
                )}

                <div className="space-y-2">
                  <Label htmlFor="title">Website Title *</Label>
                  <Input
                    id="title"
                    type="text"
                    placeholder="Enter your website title"
                    value={formData.title}
                    onChange={(e) => handleInputChange("title", e.target.value)}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="url">Website URL *</Label>
                  <Input
                    id="url"
                    type="url"
                    placeholder="https://example.com"
                    value={formData.url}
                    onChange={(e) => handleInputChange("url", e.target.value)}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="imageUrl">Website Image (Optional)</Label>
                  <Input
                    id="imageUrl"
                    type="url"
                    placeholder="https://example.com/image.jpg"
                    value={formData.imageUrl}
                    onChange={(e) => handleInputChange("imageUrl", e.target.value)}
                  />
                  <p className="text-sm text-muted-foreground">
                    Provide a URL to an image that represents your website (logo, screenshot, etc.)
                  </p>
                </div>

                <div className="space-y-4">
                  <Label className="text-base font-medium">Crawling Preferences</Label>

                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="supportWebCrawl"
                      checked={formData.supportWebCrawl}
                      onCheckedChange={(checked) => handleInputChange("supportWebCrawl", checked)}
                    />
                    <Label htmlFor="supportWebCrawl" className="text-sm font-normal">
                      Allow web crawling (recommended)
                    </Label>
                  </div>
                  <p className="text-sm text-muted-foreground ml-6">
                    This allows our search engine to index your website content for better search results.
                  </p>

                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="supportImageCrawl"
                      checked={formData.supportImageCrawl}
                      onCheckedChange={(checked) => handleInputChange("supportImageCrawl", checked)}
                    />
                    <Label htmlFor="supportImageCrawl" className="text-sm font-normal">
                      Allow image crawling
                    </Label>
                  </div>
                  <p className="text-sm text-muted-foreground ml-6">
                    This allows our search engine to index images from your website for image search results.
                  </p>
                </div>

                <div className="border-t pt-6">
                  <div className="flex items-start space-x-2">
                    <Checkbox
                      id="ownerConfirmed"
                      checked={formData.ownerConfirmed}
                      onCheckedChange={(checked) => handleInputChange("ownerConfirmed", checked)}
                      required
                    />
                    <Label htmlFor="ownerConfirmed" className="text-sm font-normal leading-relaxed">
                      I confirm that I am the owner or developer of this website and have the authority to submit it for
                      indexing.
                    </Label>
                  </div>
                </div>

                <Button type="submit" className="w-full" disabled={isSubmitting || !formData.ownerConfirmed}>
                  {isSubmitting ? (
                    <>
                      <Upload className="w-4 h-4 mr-2 animate-spin" />
                      Submitting...
                    </>
                  ) : (
                    <>
                      <Upload className="w-4 h-4 mr-2" />
                      Submit Website
                    </>
                  )}
                </Button>
              </form>
            </CardContent>
          </Card>

          <div className="mt-8 text-center text-sm text-muted-foreground">
            <p>
              By submitting your website, you agree to our terms of service and privacy policy. We review all
              submissions manually and will only index appropriate content.
            </p>
          </div>
        </div>
      </main>
    </div>
  )
}
